<?php
/* --------------------------------------------------------------
  ContentValueObjectFactory.php 2019-07-31
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/

/**
 * Class ContentValueObjectFactory
 */
class ContentValueObjectFactory
{
    /**
     * @var LanguageProviderInterface
     */
    protected $languageProvider;
    
    
    /**
     * ContentValueObjectFactory constructor.
     *
     * @param LanguageProviderInterface $languageProvider
     */
        public function __construct(LanguageProviderInterface $languageProvider)
    {
        $this->languageProvider = $languageProvider;
    }
    
    /**
     * @return LanguageProviderInterface
     */
    public function languageProvider(): LanguageProviderInterface
    {
        return  $this->languageProvider;
    }
    
    
    /**
     * @param int $languageId
     *
     * @return LanguageCode
     */
    public function getLanguageCodeFromLanguageId(int $languageId): LanguageCode
    {
        return $this->languageProvider()->getCodeById(new IdType($languageId));
    }
    
    
    /**
     * @param array $dbContent
     *
     * @return array
     * @throws Exception
     */
    public function createValueObjectsForInfoElementContent(array $dbContent): array
    {
        $position = ElementPosition::createFromString(ElementPositionMapper::getElementPositionFromDatabase($dbContent[0]['content_position']));
        $headings = $this->createContentHeadingCollection($dbContent);
        $id       = (int)$dbContent[0]['content_group'];
        $alias    = $dbContent[0]['content_alias'];
        $status   = $this->createContentStatus($dbContent);
        $texts    = $this->createContentTextCollection($dbContent);
        $titles   = $this->createContentTitleCollection($dbContent);
        
        return [
            'position' => $position,
            'headings' => $headings,
            'id'       => $id,
            'alias'    => $alias,
            'status'   => $status,
            'texts'    => $texts,
            'titles'   => $titles,
        ];
    }
    
    
    /**
     * @param array $dbContent
     * @param array $urlRewriteData
     *
     * @return array
     * @throws Exception
     */
    public function createValueObjectsForInfoPageContent(array $dbContent, array $urlRewriteData): array
    {
        $heading          = $this->createContentHeadingCollection($dbContent);
        $texts            = $this->createContentTextCollection($dbContent);
        $position         = PagePosition::createFromString(PagePositionMapper::getPagePositionFromDatabase($dbContent[0]['content_position']));
        $id               = (int)$dbContent[0]['content_group'];
        $alias            = $dbContent[0]['content_alias'];
        $status           = $this->createContentStatus($dbContent);
        $titles           = $this->createContentTitleCollection($dbContent);
        $allowRobots      = $this->createContentAllowRobotsStatus($dbContent);
        $downloadFiles    = $this->createDownloadFileCollection($dbContent);
        $metaDescriptions = $this->createContentMetaDescriptionCollection($dbContent);
        $metaKeyword      = $this->createContentMetaKeywordsCollection($dbContent);
        $metaTitle        = $this->createContentMetaTitleCollection($dbContent);
        $names            = $this->createContentNameCollection($dbContent);
        $openGraph        = $this->createContentOpenGraphImageCollection($dbContent);
        $urlKeywords      = $this->createContentUrlKeywordsCollection($dbContent);
        $urlRewrite       = $this->createContentUrlRewriteCollection($urlRewriteData);
        $sitemap          = $this->createContentSitemap($dbContent);
        
        return [
            'headings'         => $heading,
            'texts'            => $texts,
            'position'         => $position,
            'id'               => $id,
            'alias'            => $alias,
            'status'           => $status,
            'titles'           => $titles,
            'allowRobots'      => $allowRobots,
            'downloadFiles'    => $downloadFiles,
            'metaDescriptions' => $metaDescriptions,
            'metaKeywords'     => $metaKeyword,
            'metaTitles'       => $metaTitle,
            'names'            => $names,
            'openGraph'        => $openGraph,
            'urlKeywords'      => $urlKeywords,
            'urlRewrites'      => $urlRewrite,
            'sitemap'          => $sitemap,
        ];
    }
    
    
    /**
     * @param array $dbContent
     *
     * @return array
     * @throws Exception
     */
    public function createValueObjectsForLinkPageContent(array $dbContent): array
    {
        $names        = $this->createContentNameCollection($dbContent);
        $position     = PagePosition::createFromString(PagePositionMapper::getPagePositionFromDatabase($dbContent[0]['content_position']));
        $titles       = $this->createContentTitleCollection($dbContent);
        $links        = $this->createContentLinkCollection($dbContent);
        $openInNewTab = $this->createContentOpenInNewTabStatus($dbContent);
        $status       = $this->createContentStatus($dbContent);
        $id           = (int)$dbContent[0]['content_group'];
        $alias        = (int)$dbContent[0]['content_alias'];
        
        return [
            'names'        => $names,
            'position'     => $position,
            'titles'       => $titles,
            'links'        => $links,
            'openInNewTab' => $openInNewTab,
            'status'       => $status,
            'id'           => $id,
            'alias'        => $alias,
        ];
    }
    
    
    /**
     * @param array $dbContent
     *
     * @return ContentSitemap
     */
    public function createContentSitemap(array $dbContent): ContentSitemap
    {
        $contentSitemapPriorityStd = new stdClass;
        
        foreach ($dbContent as $content) {
            
            $languageCode = $this->getLanguageCodeFromLanguageId((int)$content['languages_id']);
            
            $contentSitemapPriorityStd->{$languageCode->asString()} = $content['gm_priority'];
        }
        
        $firstContent = current($dbContent);
        
        $contentSitemapVisibility         = new ContentSitemapVisibility((bool)$firstContent['gm_sitemap_entry']);
        $contentSitemapPriorityCollection = ThemeContentsParser::createLocalizedCollection($contentSitemapPriorityStd,
                                                                                           new StringType(ContentSitemapPriority::class));
        $contentSitemapChangeFrequency    = ContentSitemapChangeFrequency::createFromString($firstContent['gm_changefreq']);
        
        return new ContentSitemap($contentSitemapVisibility,
                                  $contentSitemapPriorityCollection,
                                  $contentSitemapChangeFrequency);
    }
    
    
    /**
     * @param array $dbContent
     *
     * @return ContentOpenInNewTabStatus
     */
    public function createContentOpenInNewTabStatus(array $dbContent): ContentOpenInNewTabStatus
    {
        $firstContent = current($dbContent);
        
        return new ContentOpenInNewTabStatus($firstContent['gm_link_target'] === '_blank');
    }
    
    
    /**
     * @param array $dbContent
     *
     * @return ContentLinkCollection
     */
    public function createContentLinkCollection(array $dbContent): ContentLinkCollection
    {
        $urlRewriteStd = new stdClass;
        
        foreach ($dbContent as $content) {
            
            $languageCode = $this->getLanguageCodeFromLanguageId((int)$content['languages_id']);
            
            $urlRewriteStd->{$languageCode->asString()} = $content['gm_link'];
        }
        
        return ThemeContentsParser::createLocalizedCollection($urlRewriteStd, new StringType(ContentLink::class));
    }
    
    
    /**
     * @param array $dbContent
     *
     * @return ContentUrlRewriteCollection
     */
    public function createContentUrlRewriteCollection(array $dbContent): ContentUrlRewriteCollection
    {
        $urlRewriteStd = new stdClass;
        
        foreach ($dbContent as $content) {
            
            $languageCode = $this->getLanguageCodeFromLanguageId((int)$content['language_id']);
            
            $urlRewriteStd->{$languageCode->asString()} = $content['rewrite_url'];
        }
        
        return ThemeContentsParser::createLocalizedCollection($urlRewriteStd, new StringType(ContentUrlRewrite::class));
    }
    
    
    /**
     * @param array $dbContent
     *
     * @return ContentUrlKeywordsCollection
     */
    public function createContentUrlKeywordsCollection(array $dbContent): ContentUrlKeywordsCollection
    {
        $urlKeywordsStd = new stdClass;
        
        foreach ($dbContent as $content) {
            
            $languageCode = $this->getLanguageCodeFromLanguageId((int)$content['languages_id']);
            
            $urlKeywordsStd->{$languageCode->asString()} = $content['gm_url_keywords'];
        }
        
        return ThemeContentsParser::createLocalizedCollection($urlKeywordsStd,
                                                              new StringType(ContentUrlKeywords::class));
    }
    
    
    /**
     * @param array $dbContent
     *
     * @return ContentOpenGraphImageCollection
     */
    public function createContentOpenGraphImageCollection(array $dbContent): ContentOpenGraphImageCollection
    {
        $optenGraphImageStd = new stdClass;
        
        foreach ($dbContent as $content) {
            
            $languageCode = $this->getLanguageCodeFromLanguageId((int)$content['languages_id']);
            
            $optenGraphImageStd->{$languageCode->asString()} = $content['opengraph_image'];
        }
        
        return ThemeContentsParser::createLocalizedCollection($optenGraphImageStd,
                                                              new StringType(ContentOpengraphImage::class));
    }
    
    
    /**
     * @param array $dbContent
     *
     * @return ContentNameCollection
     */
    public function createContentNameCollection(array $dbContent): ContentNameCollection
    {
        $contentNameStd = new stdClass;
        
        foreach ($dbContent as $content) {
            
            $languageCode = $this->getLanguageCodeFromLanguageId((int)$content['languages_id']);
            
            $contentNameStd->{$languageCode->asString()} = $content['content_name'];
        }
        
        return ThemeContentsParser::createLocalizedCollection($contentNameStd, new StringType(ContentName::class));
    }
    
    
    /**
     * @param array $dbContent
     *
     * @return ContentMetaTitleCollection
     */
    public function createContentMetaTitleCollection(array $dbContent): ContentMetaTitleCollection
    {
        $metaTitleStd = new stdClass;
        
        foreach ($dbContent as $content) {
            
            $languageCode = $this->getLanguageCodeFromLanguageId((int)$content['languages_id']);
            
            $metaTitleStd->{$languageCode->asString()} = $content['contents_meta_title'];
        }
        
        return ThemeContentsParser::createLocalizedCollection($metaTitleStd, new StringType(ContentMetaTitle::class));
    }
    
    
    /**
     * @param array $dbContent
     *
     * @return ContentMetaKeywordsCollection
     */
    public function createContentMetaKeywordsCollection(array $dbContent): ContentMetaKeywordsCollection
    {
        $metaKeywordsStd = new stdClass;
        
        foreach ($dbContent as $content) {
            
            $languageCode = $this->getLanguageCodeFromLanguageId((int)$content['languages_id']);
            
            $metaKeywordsStd->{$languageCode->asString()} = $content['contents_meta_keywords'];
        }
        
        return ThemeContentsParser::createLocalizedCollection($metaKeywordsStd,
                                                              new StringType(ContentMetaKeywords::class));
    }
    
    
    /**
     * @param array $dbContent
     *
     * @return ContentMetaDescriptionCollection
     */
    public function createContentMetaDescriptionCollection(array $dbContent): ContentMetaDescriptionCollection
    {
        $metaDescriptionStd = new stdClass;
        
        foreach ($dbContent as $content) {
            
            $languageCode = $this->getLanguageCodeFromLanguageId((int)$content['languages_id']);
            
            $metaDescriptionStd->{$languageCode->asString()} = $content['contents_meta_description'];
        }
        
        return ThemeContentsParser::createLocalizedCollection($metaDescriptionStd,
                                                              new StringType(ContentMetaDescription::class));
    }
    
    
    /**
     * @param array $dbContent
     *
     * @return ContentDownloadFileCollection
     */
    public function createDownloadFileCollection(array $dbContent): ContentDownloadFileCollection
    {
        $downloadFileStd = new stdClass;
        
        foreach ($dbContent as $content) {
            
            $languageCode = $this->getLanguageCodeFromLanguageId((int)$content['languages_id']);
            
            $downloadFileStd->{$languageCode->asString()} = $content['download_file'];
        }
        
        return ThemeContentsParser::createLocalizedCollection($downloadFileStd,
                                                              new StringType(ContentDownloadFile::class));
    }
    
    
    /**
     * @param array $dbContent
     *
     * @return ContentAllowRobotsStatus
     */
    public function createContentAllowRobotsStatus(array $dbContent): ContentAllowRobotsStatus
    {
        $firstContent = current($dbContent);
        
        return new ContentAllowRobotsStatus((bool)$firstContent['gm_robots_entry']);
    }
    
    
    /**
     * @param array $dbContent
     *
     * @return ContentHeadingCollection
     */
    public function createContentHeadingCollection(array $dbContent): ContentHeadingCollection
    {
        $headingStd = new stdClass;
        
        foreach ($dbContent as $content) {
            
            $languageCode = $this->getLanguageCodeFromLanguageId((int)$content['languages_id']);
            
            $headingStd->{$languageCode->asString()} = $content['content_heading'];
        }
        
        return ThemeContentsParser::createLocalizedCollection($headingStd, new StringType(ContentHeading::class));
    }
    
    
    /**
     * @param array $dbContent
     *
     * @return ContentStatus
     */
    public function createContentStatus(array $dbContent): ContentStatus
    {
        $firstContent = current($dbContent);
        
        return new ContentStatus((bool)$firstContent['content_status']);
    }
    
    
    /**
     * @param array $dbContent
     *
     * @return ContentTextCollection
     */
    public function createContentTextCollection(array $dbContent): ContentTextCollection
    {
        $textStd = new stdClass;
        
        foreach ($dbContent as $content) {
            
            $languageCode = $this->getLanguageCodeFromLanguageId((int)$content['languages_id']);
            
            $textStd->{$languageCode->asString()} = $content['content_text'];
        }
        
        return ThemeContentsParser::createLocalizedCollection($textStd, new StringType(ContentText::class));
    }
    
    
    /**
     * @param array $dbContent
     *
     * @return ContentTitleCollection
     */
    public function createContentTitleCollection(array $dbContent): ContentTitleCollection
    {
        $titleStd = new stdClass;
        
        foreach ($dbContent as $content) {
            
            $languageCode = $this->getLanguageCodeFromLanguageId((int)$content['languages_id']);
            
            $titleStd->{$languageCode->asString()} = $content['content_title'];
        }
        
        return ThemeContentsParser::createLocalizedCollection($titleStd, new StringType(ContentTitle::class));
    }
}